<?php

namespace App\Http\Controllers;

use App\Http\Requests;
use App\Http\Controllers\Controller;

use App\Customer;
use App\Address;
use App\CustomerInvoice;
use App\CustomerInvoiceData;
use Illuminate\Http\Request;
use Session, Datatables;
use Carbon\Carbon;

class CustomersController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index()
    {
        $title = 'Customers';
        return view('customers.index', compact('title'));
    }
    
    public function getCustomerData(Request $request)
    {
        $type = $request['type'];
        $where['user_id'] = userId();
        
        $customer = Customer::with('address')->where($where);
        
        return Datatables::eloquent($customer, $type)
            ->editColumn('id', function ($customer) use($type) {
                if($type == 1)
                {
                    return $customer->id;
                }else if($type == 2)
                {
                    return '<a href="javascript:;" class="customer_details" id="'.$customer->id.'|'.$customer->title.' '.$customer->first_name.' '.$customer->last_name.'">'.$customer->id.'</a>';
                }
            })    
            ->addColumn('action', function($customer) {
                return '<a href="'.url('/customers/'.$customer->id.'/edit').'" class="ajaxify"><i class="fa fa-edit"></i></a>';
            })->make(true);
    }
    
    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $title    = 'Customers';
        $subTitle = 'Add Customer';
        
        return view('customers.create', compact('title', 'subTitle'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'email'          => 'email|unique:customers',
            'title'          => 'required|alpha|max:5',
            'first_name'     => 'required|alpha_dash_space|max:50',
            'last_name'      => 'required|alpha_dash_space|max:50',
            'address_line_1' => 'max:255',
            'town'           => 'max:50',
            'county'         => 'max:50',
            'post_code'      => 'max:20',
            'phone_no'       => 'required|digits_between:1,20',
        ]);
        
        $request['user_id'] = userId();
        $requestData = $request->all();
        
        $customer = Customer::create($requestData);
        $customerId = $customer->id;
        
        $request['country_id'] = 230;
        $request['default_address'] = 1;
        $request['customer_id'] = $customerId;
        Address::create($request->all());
        $customerName = $customer->title.' '.$customer->first_name.' '.$customer->last_name;
              
        if($request['module'] != '')
        {
            return response()->json(['response' => 'success', 'module' => $request['module'], 'customer_id' => $customerId, 'customer_name' => $customerName]);
        }else
        {
            Session::flash('flash_message', 'Customer added!');
            return response()->json(['response' => 'success']);
        }
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $title    = 'Customers';
        $subTitle = 'Edit Customer';
        
        $where['user_id'] = userId();
        
        $customer = Customer::where($where)->findOrFail($id);
        $address = $customer->address;
        
        return view('customers.edit', compact('title', 'subTitle', 'customer', 'address'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  int  $id
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update($id, Request $request)
    {
        $this->validate($request, [
            'email'          => 'email|unique:customers,email,'.$id,
            'title'          => 'required|alpha|max:5',
            'first_name'     => 'required|alpha_dash_space|max:50',
            'last_name'      => 'required|alpha_dash_space|max:50',
            'address_line_1' => 'max:255',
            'town'           => 'max:50',
            'county'         => 'max:50',
            'post_code'      => 'max:20',
            'phone_no'       => 'required|digits_between:1,20',
        ]);
        
        $requestData = $request->all();
        
        $where['user_id'] = userId();
        $customer = Customer::where($where)->findOrFail($id);
        $customer->update($requestData);

        $addressId = $request['address_id'];
        $address = Address::where($where)->findOrFail($addressId);
        $address->update($request->except(['address_id']));
        
        Session::flash('flash_message', 'Customer updated!');

        return response()->json(['response' => 'success']);
    }
/*    
    public function updateStatus(Request $request)
    {
        $id = $request['id'];
        $status = $request['status'];
        
        $where['id'] = $id;
        $where['user_id'] = userId();
        
        $result = Customer::where($where)->update('status', $status);

        if($result > 0)
        {
            if($status == 1)
            {
                $message = 'Status enabled!';
            }else
            {
                $message = 'Status disabled!';
            }
            
            Session::flash('flash_message', $message);
            return response()->json(['response' => 'success']);
        }else
        {
            $message = 'Status not updated!';
            
            return response()->json(['message' => $message]);
        }
    }
*/    
    public function customerInvoice(Request $request)
    {
        $bundleData = '';
        $costPrice  = 0.00;
        $title      = 'Generate Customer Invoice';
        
        $customerId   = $request['customer_id'];
        $customerName = $request['customer_name'];
        $moduleId     = $request['module_id'];
        $tableId      = $request['id'];
         
        if($moduleId == 5 || $moduleId == 6)
        {
            $where['id'] = $tableId;
            $where['user_id'] = userId();
            
            $parcel = \App\ParcelInformation::where($where)->first();
            
            $bundleData = $parcel->bundle_data;
            
            if($parcel->type == 1)
            {
               $costPrice = $parcel->courierService->total_price;
            }else if($parcel->type == 2)
            {
                $costPrice = $parcel->cargo->total;
            }
        }
        
        return view('customers.customer_invoice', compact('title', 'customerId', 'customerName', 'tableId', 'moduleId', 'costPrice', 'bundleData'));
    }
    
    public function customerInvoiceStore(Request $request)
    {        
        $moduleId = $request['module_id'];
        
        $validation['customer_id'] = 'required';
        $validation['cost_price']  = 'required|numeric';
        $validation['sale_price']  = 'required|numeric';
        $validation['table_id']    = 'required_if:module_id,5,6';
        $validation['module_id']   = 'required';
        
        if($moduleId == 1 || $moduleId == 7 || $moduleId == 14)
        {
            $validation['cost_price'] = 'required|numeric|max:'.getClientLimit();
        }
        
        $this->validate($request, $validation);
        
        if($moduleId == 1 || $moduleId == 7 || $moduleId == 14)
        {
            $transactionId = getAgentTransactionId();
            if($moduleId == 1)
            {
                $costPrice = getGlobalValue('e-attractions_commission');
            }else if($moduleId == 7)
            {
                $costPrice = getGlobalValue('e-hotels_commission');
            }else if($moduleId == 14)
            {
                $costPrice = getGlobalValue('e-tours_commission');
            }
            
            $request['cost_price'] = $costPrice;
            $request['agent_transaction_id'] = $transactionId;
        }
        
        if($moduleId == 13)
        {
            $request['status'] = 0;
        }
        
        if($moduleId != 13)
        {
            $request['transaction_id'] = getTransactionId();
        }
        
        $request['user_id'] = userId();
        $requestData = $request->all();
        
        $customerInvoice = CustomerInvoice::create($requestData);
        $customerInvoiceId = $customerInvoice->id;
        
        if($moduleId == 1 || $moduleId == 7 || $moduleId == 14)
        {
            $data['amount']  = $costPrice;
            $data['action']  = $customerInvoice->module->name.' Commission (Transaction ID: '.$transactionId.')';
            $data['user_id'] = userId();

            \App\UserBalance::create($data);

            $where['id'] = userId();

            \DB::table('users')->where($where)->decrement('balance', $costPrice);
        }
        
        $data = array();
        
        if($request['field_id'] != '')
        {
            foreach($request['field_id'] as $key => $value):
                $fieldValue = isset($request['field_name'][$key])?$request['field_name'][$key]:'';

                if($fieldValue != '')
                {
                    $data[] = array(
                            'custom_field_id' => $value,
                            'field_value'     => $fieldValue,
                            'invoice_id'      => $customerInvoiceId,
                            'user_id'         => userId(),
                            'created_at'      => date('Y-m-d H:i:s'),
                            'updated_at'      => date('Y-m-d H:i:s')
                        );
                }
            endforeach;
        }
        
        if(!empty($data))
        {
            CustomerInvoiceData::Insert($data);
        }
        if($request['module_id'] == 13)
        {
            Session::flash('flash_message', 'Quote is generated');
        }  else 
        {
            Session::flash('flash_message', 'Invoice is generated');
        }
        
        Session::flash('customer_invoice_id', $customerInvoiceId);
        return response()->json(['response' => 'success']);
    }
    
    public function quoteToInvoice(Request $request)
    {
        if(count(array_filter($request['field_name'])) > 0)
        {
            $this->validate($request, [
                'quote_id' => 'required'
            ]);

            $quoteId = $request['quote_id'];

            $where['id'] = $quoteId;
            $where['user_id'] = userId();

            CustomerInvoice::where($where)->update(array('transaction_id' => getTransactionId(), 'status' => 1));

            $data = array();

            if($request['field_id'] != '')
            {
                foreach($request['field_id'] as $key => $value):
                    $fieldValue = isset($request['field_name'][$key])?$request['field_name'][$key]:'';

                    if($fieldValue != '')
                    {
                        $data[] = array(
                                'custom_field_id' => $value,
                                'field_value'     => $fieldValue,
                                'invoice_id'      => $quoteId,
                                'user_id'         => userId(),
                                'created_at'      => date('Y-m-d H:i:s'),
                                'updated_at'      => date('Y-m-d H:i:s')
                            );
                    }
                endforeach;
            }

            if(!empty($data))
            {
                CustomerInvoiceData::Insert($data);
            }

            Session::flash('flash_message', 'Invoice is generated');
            Session::flash('customer_invoice_id', $quoteId);

            return response()->json(['response' => 'success']);
        }else{
            return response()->json(['message' => 'Please fill at least one field!']);
        }
    }
    
    public function customerReceipt(Request $request)
    {
        $parcelInformation = '';
         
        $where['id'] = $request['customer_invoice_id'];
        $where['user_id'] = userId();
        
        $customerInvoice = CustomerInvoice::where($where)->first();
        
        $where_1['customer_invoice_data.user_id'] = userId();
        $where_1['invoice_id'] = $request['customer_invoice_id'];
        
        $customerInvoiceData = CustomerInvoiceData::select('name', 'field_type_id', 'field_values', 'field_value')
                                    ->join('custom_fields', function ($join) {
                                        $join->on('customer_invoice_data.custom_field_id', '=', 'custom_fields.id')
                                                ->where('after_amount', 0);
                                    })->where($where_1)->orderBy('sort_order')->get();
        
        $customerInvoiceData2 = CustomerInvoiceData::
                                    join('custom_fields', function ($join) {
                                        $join->on('customer_invoice_data.custom_field_id', '=', 'custom_fields.id')
                                                ->where('after_amount', 1);
                                    })->where($where_1)->orderBy('sort_order')->get();
        
        if($customerInvoice->module_id == 5 || $customerInvoice->module_id == 6)
        {
            unset($where);
            $where['id'] = $customerInvoice->table_id;
            $where['user_id'] = userId();
            
            $parcelInformation = \App\ParcelInformation::where($where)->first();
        }
        
        return view('customers.customer_receipt', compact('customerInvoice', 'customerInvoiceData', 'customerInvoiceData2', 'parcelInformation'));
    }
    
    public function getCustomData(Request $request)
    {
        $where['module_id'] = $request['module_id'];
        $where['modal']     = $request['modal'];
        $where['user_id']   = userId();
        
        $customFields = \App\CustomField::where($where)->orderBy('sort_order')->get();
        
        return view('customers.custom_field_data', compact('customFields'));
    }
    
    public function getQuotes(Request $request)
    {
        $date = date('Y-m-d H:i:s');
        
        $dt = Carbon::parse($date);
        $dateFrom = $dt->subMonth();
                
        $where['module_id'] = 13;
        $where['status'] = 0;
        $where['customer_id'] = $request['customer_id'];
        $where['user_id'] = userId();

        $customerInvoice = CustomerInvoice::where($where)->whereBetween('created_at', array($dateFrom, $date))->orderBy('created_at', 'desc')->get();

        return view('customers.quotes_lookup_modal', compact('customerInvoice'));
    }
}
